# quickgen_metrics_ui.R
library(shinyjs)
library(shinyAce)

## ==================== 右下角浮框 ====================
ui_metrics_progress <- tags$div(
  id = "metrics_progress_box",
  style = "display:none;
           position:fixed;
           bottom:15px; right:15px;
           width:220px; z-index:9999;
           background:#f5f5f5; color:#333;
           border:1px solid #337ab7; border-radius:4px;
           padding:10px 15px; box-shadow:0 2px 8px rgba(0,0,0,.25);",
  tags$strong(i18n$t("AI generating...")),
  tags$div(class = "progress",
           tags$div(class = "progress-bar progress-bar-striped active",
                    style = "width:100%"))
)
## ======== 警告弹窗========
ui_metrics_warn <- tags$div(
  id = "metrics_warn_box",
  style = "display:none;
           position:fixed;
           bottom:15px; right:15px;
           width:220px; z-index:9999;
           background:#fff3cd; color:#856404;
           border:1px solid #ffeaa7; border-radius:4px;
           padding:10px 15px; box-shadow:0 2px 8px rgba(0,0,0,.25);",
  tags$strong(i18n$t("Warning:Please enter a request related to scientific research statistical calculation."))
)
## ==================== 统一入口 ====================
output$quickgen_metrics <- renderUI({
  stat_tab_panel(
    menu      = i18n$t("Oneclick generation > AI assistance"),
    tool      = i18n$t("AI generates descriptive statistics(metrics)"),
    tool_ui   = "metrics_main_ui",
    output_panels = tabPanel(
      title = i18n$t("Metrics Assistant"),
      value = "metrics_panel",
      uiOutput("metrics_result_area")
    )
  )
})
## ==================== 右侧 Metrics 面板 ====================
output$metrics_main_ui <- renderUI({
  tagList(
    useShinyjs(),
    ui_metrics_progress,
    ui_metrics_warn,
    wellPanel(
      i18n$t("Describe your analysis request:"),
      returnTextAreaInput("metrics_prompt",
                          label   = NULL,
                          placeholder = i18n$t("e. g. 1. Single mean test for age (null mean=35) in insurance; 2. Chi-squared test between sex and smoker; 3. Linear regression of charges on bmi+age"),
                          rows    = 4,
                          value   = state_init("metrics_prompt", "")),
      fluidRow(
        column(6, uiOutput("ui_metrics_submit")),
        column(6, uiOutput("metrics_loading"))
      )
    ),
    
    wellPanel(
      i18n$t("Generated R code"),
      uiOutput("metrics_r_code_block"),
      fluidRow(
        column(6, actionButton("metrics_run_code", i18n$t("Run code"), icon = icon("play"), class = "btn-success")),
        column(6, actionButton("metrics_edit_code", i18n$t("Edit code"), icon = icon("edit"), class = "btn-default"))
      )
    ),
    
    help_and_report(
      modal_title = i18n$t("AI generates descriptive statistics(metrics)"),
      fun_name    = "quickgen_metrics",
      help_file   = inclMD(file.path(getOption("radiant.path.quickgen"), "app/tools/help/quickgen_metrics.md")),
      lic = "by-sa"
    )
  )
})
## ==================== 控件渲染 ====================
output$ui_metrics_submit <- renderUI({
  req(input$dataset)
  actionButton("metrics_submit", i18n$t("Send"), icon = icon("magic"), class = "btn-primary")
})
output$metrics_loading <- renderUI({
  if (isTRUE(r_values$metrics_loading))
    tags$div(class = "progress",
             tags$div(class = "progress-bar progress-bar-striped active",
                      style = "width:100%",
                      i18n$t("Calling metrics calculation model...")))
})
## ==================== reactiveValues ====================
r_values <- reactiveValues(
  metrics_r_code      = "",
  metrics_result_type = "text",  # 仅保留text/plot/error
  metrics_result_ready = FALSE,
  metrics_loading     = FALSE
)
## ==================== 生成代码 ====================
observeEvent(input$metrics_submit, {
  if (is.empty(input$metrics_prompt))
    return(showNotification(i18n$t("Please enter a statistical calculation request"), type = "error"))
  
  r_values$metrics_loading <- TRUE
  shinyjs::show("metrics_progress_box")          # 显示右下角进度框
  on.exit({
    r_values$metrics_loading <- FALSE
    shinyjs::hide("metrics_progress_box")        # 无论成功失败都隐藏
  })
  
  res <- try(do.call(metrics_generate,
                     list(prompt  = input$metrics_prompt,
                          dataset = input$dataset,
                          envir   = r_data)),
             silent = TRUE)
  if (inherits(res, "try-error")) {
    showNotification(paste(i18n$t("Metrics API error:"), res), type = "error")
    return()
  }
  
  r_values$metrics_r_code      <- res$r_code
  r_values$metrics_result_type <- res$type
  r_values$metrics_result_ready <- FALSE
  r_values$auto_run       <- res$auto_run
  
  if (res$type == "empty") {
    shinyjs::show("metrics_warn_box")
    shinyjs::delay(3000, shinyjs::hide("metrics_warn_box"))
    return()
  }
  
  if (isTRUE(r_values$auto_run))
    shinyjs::click("metrics_run_code")
})
## ==================== 显示 R 代码 ====================
output$metrics_r_code_block <- renderUI({
  codes <- r_values$metrics_r_code
  tags$pre(codes, style = "background:#f5f5f5; padding:10px; border-radius:4px;
                           font-family:monospace; white-space:pre-wrap; min-height:100px;")
})
## ==================== 运行代码 ====================
observeEvent(input$metrics_run_code, {
  shinyjs::hide("metrics_progress_box")   
  if (is.empty(r_values$metrics_r_code)) return()
  
  if (trimws(r_values$metrics_r_code) == "" || identical(r_values$metrics_result_type, "empty")) {
    shinyjs::show("metrics_warn_box")
    shinyjs::delay(3000, shinyjs::hide("metrics_warn_box"))
    return()
  }
  
  tryCatch({
    # 执行代码并获取原始结果
    result <- do.call(metrics_run_code,
                      list(r_code = r_values$metrics_r_code, envir = r_data))
    r_data$metrics_temp_result <- result
    dataset_name <- input$dataset  # 获取当前数据集名称
    
    # 分支1：屏蔽图表输出
    if (inherits(result, "gg") || inherits(result, "ggplot") || inherits(result, "lattice")) {
      r_values$metrics_result_type <- "text"
      output$metrics_result_text <- renderText(i18n$t("This tool only supports statistical metrics calculation, not chart generation"))
    } 
    # 分支2：所有统计结果转为结构化文本
    else {
      r_values$metrics_result_type <- "text"
      if (is.null(result) || result == "") {
        output$metrics_result_text <- renderText(i18n$t("No output generated."))
      } else {
        output$metrics_result_text <- renderText(result)
      }
    }
    r_values$metrics_result_ready <- TRUE
  }, error = function(e) {
    r_values$metrics_result_type <- "error"
    # 错误文本格式化
    err_msg <- if (grepl("object .* not found", e$message)) {
      paste0(i18n$t("Error: Target column does not exist - "), e$message)
    } else if (grepl("non-numeric argument", e$message)) {
      paste0(i18n$t("Error: Variable type mismatch (e.g., t-test requires numeric variables) - "), e$message)
    } else {
      paste0(i18n$t("Error: "), e$message)
    }
    output$metrics_result_error <- renderText(err_msg)
    r_values$metrics_result_ready <- TRUE
    showNotification(err_msg, type = "error", duration = NULL)
  })
}, ignoreInit = TRUE)
## ======== 结果展示区（仅文本/错误输出）========
output$metrics_result_area <- renderUI({
  req(r_values$metrics_result_ready)
  tagList(
    # 所有结果均用文本框展示（含图表屏蔽提示、统计结果、错误）
    conditionalPanel(
      condition = "output.metrics_result_type == 'text' || output.metrics_result_type == 'error'",
      verbatimTextOutput("metrics_result_text", placeholder = TRUE)
    )
  )
})
output$metrics_result_type <- reactive({
  r_values$metrics_result_type 
})
outputOptions(output, "metrics_result_type", suspendWhenHidden = FALSE)
## ==================== 编辑代码模态框 ====================
observeEvent(input$metrics_edit_code, {
  showModal(
    modalDialog(
      title = i18n$t("Edit R Code"),
      size  = "l",
      footer = tagList(
        actionButton("metrics_save_code", i18n$t("Save Changes"), class = "btn-primary"),
        modalButton(i18n$t("Cancel"))
      ),
      aceEditor(
        "metrics_code_editor",
        mode = "r",
        theme = getOption("radiant.ace_theme", "tomorrow"),
        wordWrap = TRUE,
        value = r_values$metrics_r_code,
        placeholder = i18n$t("Edit the generated R code here..."),
        vimKeyBinding = getOption("radiant.ace_vim.keys", FALSE),
        tabSize = getOption("radiant.ace_tabSize", 2),
        useSoftTabs = getOption("radiant.ace_useSoftTabs", TRUE),
        showInvisibles = getOption("radiant.ace_showInvisibles", FALSE),
        autoScrollEditorIntoView = TRUE,
        minLines = 15,
        maxLines = 30
      )
    )
  )
})
## ==================== 保存代码 ====================
observeEvent(input$metrics_save_code, {
  r_values$metrics_r_code <- input$metrics_code_editor
  r_values$auto_run <- FALSE
  removeModal()
})
## ==================== 报告 / 截图 ====================
metrics_report <- function() {}
observeEvent(input$metrics_report, metrics_report())
observeEvent(input$metrics_screenshot, radiant_screenshot_modal("modal_metrics_screenshot"))
observeEvent(input$modal_metrics_screenshot, { metrics_report(); removeModal() })
