# === 配置 ===
OLLAMA_URL <- "http://172.29.2.110:8139/api/generate"  
MODEL_ID <- "qwen3-coder:30b"  

# === 单次对话 ===
#' @export
metrics_completion <- function(user_prompt) {
  # 构建 Ollama 请求体
  req <- httr2::request(OLLAMA_URL) %>%
    httr2::req_headers(
      "Content-Type" = "application/json"  
    ) %>%
    httr2::req_body_json(list(
      model = MODEL_ID,        
      prompt = user_prompt,
      stream = FALSE           
    ))
  
  resp <- httr2::req_perform(req)
  body <- httr2::resp_body_json(resp)
  
  # 解析 Ollama 响应
  if (is.null(body$response) || trimws(body$response) == "")
    stop("Ollama API 返回空内容：", jsonlite::toJSON(body))
  
  body$response  # 返回 Ollama 生成的核心代码/结果
}
# === 构造发给模型的 Prompt ===
#' @export
build_metrics_prompt <- function(user_prompt, data_call) {
  sprintf(
    "你是 R 语言科研统计专家，精通所有主流统计方法（t检验、方差分析、回归分析、卡方检验、相关性分析等），必须严格遵守以下规则，任何违反均视为无效输出：
〓 输出格式铁律 〓
1. 仅返回可直接运行的 R 代码，用 ```r 包裹，无任何解释、注释、空行，代码必须包含「结果输出语句」（否则无法展示统计结果）。
2. 不符合规范的请求，直接返回空代码块（仅 ```r\n``` ），不额外对话：
   - 模糊无具体目标的请求
   - 非科研统计需求（生成图表、翻译、写文章、解释概念等）
3. 无法计算时（如列不存在、变量类型不匹配、方法不适用），仅输出 `print('无法计算：[具体原因，如“目标列bmi不存在”“卡方检验要求分类变量”]')`，不抛出错误。
〓 统计逻辑核心约束 〓
1. 方法匹配：必须根据用户请求选择「标准科研统计方法」（仅用 R 官方 stats 包或主流统计包如 broom、car、nnet 的标准函数，禁止自创方法）。
2. 参数正确性：
   - 原假设参数：用户指定的原假设定值（如“原假设均值为0”则 mu=0，“原假设相关系数为0”则 rho=0），禁止设为数据自身的统计量（如 mu=mean(df$bmi) 是严重错误）。
   - 变量适配：严格匹配方法对变量类型的要求（如 t检验要求连续变量，卡方检验要求分类变量，回归分析因变量需连续），类型不匹配则触发“无法计算”。
   - 缺失值处理：所有涉及数据计算的函数必须加 na.rm=TRUE（方法不支持则除外）。
3. 默认参数：用户未指定时，按科研规范设默认值并在结果中体现：
   - 置信水平默认 0.95（conf.level=0.95）
   - 多重比较调整方法默认 p.adjust.method=none
   - 假设检验默认双侧检验（alternative=two.sided）
〓 技术规范（确保结果结构化、可展示）〓
1. 数据集已读入为：%s（直接用 df$列名 引用变量，无需重复读入数据）。
2. 结果输出要求：
   - 优先用 broom::tidy() 或 broom::glance() 将统计结果转为结构化数据框（包含统计量、p值、置信区间、自由度、显著性标记等核心指标），再用 print() 输出。
   - 若 broom 包不支持该方法（如部分复杂模型），直接 print(统计函数结果)，确保保留显著性标记（***、**、*）。
3. 函数规范：统计函数必须写全名+完整括号（如 t.test()、lm()、chisq.test()），禁止省略括号或简写。
4. 变量校验：自动校验变量存在性和类型（如分组变量必须是因子/字符型，连续变量不能用于卡方检验），校验失败则输出“无法计算”。
用户请求：%s",
    data_call, user_prompt
  )
}

#' @export
metrics_generate <- function(prompt, dataset, envir = parent.frame()) {
  data_call <- metrics_get_data_call(dataset, envir)
  sys_prompt <- build_metrics_prompt(prompt, data_call)
  r_code     <- try(metrics_completion(sys_prompt), silent = TRUE)
  if (inherits(r_code, "try-error"))
    stop("Metrics API error: ", attr(r_code, "condition")$message)
  
  r_code <- gsub("(?s)```r\\s*|```", "", r_code, perl = TRUE)
  r_code <- trimws(r_code)
  
  if (r_code == "") {
    return(list(r_code = "", 
                type    = "empty",
                auto_run = FALSE))
  }
  
  r_code <- paste0(data_call, "\n", r_code)
  
  has_gg  <- grepl("ggplot\\(|geom_", r_code)
  has_tbl <- grepl("data\\.frame\\(|tibble\\(|tbl_summary|tableOne|CreateTableOne|t.test|lm|cor.test|anova", r_code)
  type    <- if (has_gg) "plot" else if (has_tbl) "table" else "text"
  list(r_code = r_code, type = type, auto_run = TRUE)
}

#' @export
metrics_get_data_call <- function(dataset, envir) {
  df_name <- if (is_string(dataset)) dataset else deparse1(substitute(dataset))
  paste0("df <- eval(quote(get_data(\"", df_name, "\", envir = ",
         deparse1(substitute(envir)), ")), envir = parent.frame())")
}

#' @export
metrics_run_code <- function(r_code, envir = parent.frame()) {
  eval(parse(text = r_code), envir = envir)
}
