# ============================================================
# [新增功能] 企业级集成：Token鉴权 + 环境变量 + 自动加载
# ============================================================
observe({
  # 1. 解析 URL 参数
  query <- parseQueryString(session$clientData$url_search)

  dataset_id <- query[['datasetId']]
  token <- query[['token']]

  # 初始化一个 session 级别的变量来记录已加载的 ID，防止刷新时的重复请求
  if (is.null(session$userData$loaded_datasets)) {
    session$userData$loaded_datasets <- c()
  }

  # 2. 仅当 ID 和 Token 均存在时执行
  if (!is.null(dataset_id) && !is.null(token)) {

    # 3. [修改点] 检查 ID 是否已在本次会话中加载过
    # 因为现在数据集名称是动态的 Sheet 名，下载前无法预知，所以改为判断 ID
    if (!(dataset_id %in% session$userData$loaded_datasets)) {

      withProgress(message = '正在从业务系统同步数据...', value = 0.2, {

        # 4. 获取环境变量
        api_base <- Sys.getenv("HOST_API_BASE", "https://ds.cixincloud.com/data-search-api")

        # 5. 拼接完整 API 路径
        target_url <- paste0(api_base, "/research-project/generate-project-dataset/", dataset_id)

        # 6. 创建临时文件
        tmp_file <- tempfile(fileext = ".xlsx")

        tryCatch({
          incProgress(0.3, detail = "正在鉴权并下载...")

          # 7. HTTP 请求
          response <- httr::POST(
            url = target_url,
            httr::add_headers(authentication = paste(token)),
            httr::write_disk(tmp_file, overwrite = TRUE)
          )

          message("=== Request Log ===")
          print(response$request)
          message("===================")

          if (httr::status_code(response) != 200) {
            stop(paste("下载失败，HTTP状态码:", httr::status_code(response)))
          }

          incProgress(0.7, detail = "解析并导入 Radiant...")

          # 8. 调用核心加载函数
          # 注意：这里的 fname 仅用于扩展名识别，实际数据集名称由 Sheet 名决定
          load_user_data(
            fname = "downloaded_data.xlsx",
            uFile = tmp_file,
            ext = "xlsx",
            xlsx_header = TRUE,
            man_str_as_factor = TRUE
          )

          # 9. 界面联动：获取刚才加载的 Sheet 名称并选中第一个
          # 必须再次读取 Sheet 列表并做 make.names 处理，才能匹配到 r_data 中的 Key
          loaded_sheets <- try(readxl::excel_sheets(tmp_file), silent = TRUE)

          if (!inherits(loaded_sheets, "try-error") && length(loaded_sheets) > 0) {

            # 必须和 load_user_data 里的逻辑一致，使用 make.names 清洗
            valid_names <- make.names(loaded_sheets)
            first_sheet_name <- valid_names[1] # 默认选第一个

            # 标记为已加载
            session$userData$loaded_datasets <- c(session$userData$loaded_datasets, dataset_id)

            # 更新下拉框并选中第一个 Sheet
            updateSelectInput(session, "dataset", selected = first_sheet_name)
            updateTabsetPanel(session, "nav_radiant", selected = "Data")
            updateTabsetPanel(session, "tabs_data", selected = "View")

            showNotification(paste0("成功加载 ", length(valid_names), " 个数据集 (ID: ", dataset_id, ")"), type = "message")
          } else {
            showNotification("文件下载成功，但未发现有效的 Sheet", type = "warning")
          }

        }, error = function(e) {
          showNotification(paste("数据同步失败:", e$message), type = "error", duration = 10)
        })
      })
    }
  }
})
# ============================================================
