#' Cox Proportional Hazards Regression (minimal)
#'
#' @export
coxp <- function(dataset,
                 time,
                 status,
                 evar,
                 int = "",
                 check = "",
                 form,
                 data_filter = "",
                 arr = "",
                 rows = NULL,
                 envir = parent.frame()) {
  
  ## ---- 公式入口 ----------------------------------------------------------
  if (!missing(form)) {
    form  <- as.formula(format(form))
    vars  <- all.vars(form)
    time  <- vars[1]
    status<- vars[2]
    evar  <- vars[-(1:2)]
  }
  
  ## ---- 基础检查 ----------------------------------------------------------
  if (time %in% evar || status %in% evar) {
    return("Time/status variable contained in explanatory variables." %>%
             add_class("coxp"))
  }
  
  vars <- unique(c(time, status, evar))
  df_name <- if (is_string(dataset)) dataset else deparse(substitute(dataset))
  dataset <- get_data(dataset, vars, filt = data_filter, arr = arr, rows = rows, envir = envir)
  
  ## ---- 构造公式 ----------------------------------------------------------
  if (missing(form)) {
    rhs <- if (length(evar) == 0) "1" else paste(evar, collapse = " + ")
    if (!is.empty(int)) rhs <- paste(rhs, paste(int, collapse = " + "), sep = " + ")
    form <- as.formula(paste("Surv(", time, ", ", status, ") ~ ", rhs))
  }
  
  ## ---- 模型估计 ----------------------------------------------------------
  if ("robust" %in% check) {
    model <- survival::coxph(form, data = dataset, robust = TRUE)
  } else {
    model <- survival::coxph(form, data = dataset)
  }
  
  ## ---- 打包返回 ----------------------------------------------------------
  out <- as.list(environment())
  out$model   <- model
  out$df_name <- df_name
  out$type    <- "survival"
  out$check   <- check
  add_class(out, c("coxp", "model"))
}

#' Summary 占位
#' @export
summary.coxp <- function(object, ...) {
  if (is.character(object)) return(object)
  summary(object$model)
}

#' Predict 占位
#' @export
predict.coxp <- function(object, pred_data = NULL, pred_cmd = "",
                         dec = 3, envir = parent.frame(), ...) {
  if (is.character(object)) return(object)
  ## 如需生存预测，可返回 linear.predictors 或 survival 曲线
  pfun <- function(m, newdata) predict(m, newdata = newdata, type = "lp")
  predict_model(object, pfun, "coxp.predict",
                pred_data, pred_cmd, dec = dec, envir = envir)
}

#' Print 预测占位
#' @export
print.coxp.predict <- function(x, ..., n = 10) {
  print_predict_model(x, ..., n = n, header = "Cox Proportional Hazards")
}