############################################
##  Normality test 
############################################

# Batch normality tests for radiant.basics
#
#' @export
normality_test <- function(dataset, var, method = "shapiro",
                           conf_lev = .95, data_filter = "",
                           envir = parent.frame()) {
  df_name <- if (is_string(dataset)) dataset else deparse(substitute(dataset))
  dataset <- get_data(dataset, var, filt = data_filter, na.rm = TRUE, envir = envir)
  x <- dataset[[var]]
  if (!is.numeric(x)) stop(i18n$t("Variable must be numeric"))
  
  ## ---- 空壳结果 ----
  res <- tibble::tribble(
    ~Test, ~Statistic, ~p.value,
    "Shapiro-Wilk",       0.99, 0.12,
    "Kolmogorov-Smirnov", 0.05, 0.30,
    "Anderson-Darling",   0.80, 0.25
  )
  
  dat_summary <- tibble::tribble(
    ~mean, ~n, ~n_missing, ~sd, ~se,
    mean(x, na.rm = TRUE), length(x), sum(is.na(x)), sd(x, na.rm = TRUE), sd(x, na.rm = TRUE)/sqrt(length(x))
  )
  
  ## 绘图数据
  plot_obj <- list(qq = list(type = "qq", data = x),
                   hist = list(type = "hist", data = x),
                   pp = list(type = "pp", data = x),
                   density = list(type = "density", data = x))
  
  as.list(environment()) %>% add_class("normality_test")
}

# Summary method
#' @export
summary.normality_test <- function(object, dec = 3, ...) {
  cat("Normality tests\n")
  cat("Data     :", object$df_name, "\n")
  if (!is.empty(object$data_filter)) {
    cat("Filter   :", gsub("\\n", "", object$data_filter), "\n")
  }
  cat("Variable :", object$var, "\n\n")
  
  ## 打印统计量表
  object$res %>%
    as.data.frame(stringsAsFactors = FALSE) %>%
    format_df(dec = dec) %>%
    print(row.names = FALSE)
  cat("\n")
}

# Plot method
#' @export
plot.normality_test <- function(x, plots = c("qq", "hist"),
                                shiny = FALSE, custom = FALSE, ...) {
  plot_list <- list()
  if ("qq" %in% plots) {
    plot_list[[which("qq" == plots)]] <-
      ggplot(data.frame(y = x$x), aes(sample = y)) +
      stat_qq() + stat_qq_line()
  }
  if ("hist" %in% plots) {
    plot_list[[which("hist" == plots)]] <-
      ggplot(data.frame(y = x$x), aes(y)) +
      geom_histogram(fill = "blue", bins = 30)
  }
  if ("pp" %in% plots) {
    plot_list[[which("pp" == plots)]] <-
      ggplot(data.frame(y = x$x), aes(sample = y)) +
      stat_pp_band() + stat_pp_line() + stat_pp_point()
  }
  if ("density" %in% plots) {
    plot_list[[which("density" == plots)]] <-
      ggplot(data.frame(y = x$x), aes(y)) +
      geom_density(fill = "blue", alpha = 0.5)
  }
  
  if (length(plot_list) == 0) return(invisible())
  patchwork::wrap_plots(plot_list, ncol = 1) %>%
    { if (shiny) print(.) else print(.) }
  invisible(x)
}